//
//  LifeBoostView.swift
//  death_app Watch App
//
//  Created by Task Master on 2025-09-16.
//

import SwiftUI

struct LifeBoostView: View {
    let severity: Double // 0.0 to 1.0
    
    @State private var isAnimating = false
    @State private var sparkleOffset: CGFloat = 0
    @State private var glowIntensity: Double = 0.5
    
    private var boostColor: Color {
        Color.green.opacity(0.7 + (severity * 0.3))
    }
    
    private var scale: Double {
        1.0 + (severity * 0.3) // 1.0 to 1.3 based on severity
    }
    
    var body: some View {
        VStack(spacing: 4) {
            ZStack {
                // Central heart with glow effect
                Image(systemName: "heart.fill")
                    .font(.title2)
                    .foregroundColor(boostColor)
                    .scaleEffect(scale)
                    .shadow(color: boostColor, radius: glowIntensity * 4)
                
                // Animated sparkles around the heart
                ForEach(0..<3, id: \.self) { index in
                    sparkle(at: index)
                }
                
                // Upward arrows indicating improvement
                VStack(spacing: 1) {
                    Image(systemName: "arrow.up")
                        .font(.caption2)
                        .foregroundColor(.green)
                        .opacity(0.8)
                        .offset(y: sparkleOffset)
                    
                    Image(systemName: "arrow.up")
                        .font(.caption2)
                        .foregroundColor(.green)
                        .opacity(0.6)
                        .offset(y: sparkleOffset + 3)
                }
                .offset(x: 20)
            }
            
            Text("Improving")
                .font(.caption2)
                .foregroundColor(.green)
                .fontWeight(.medium)
        }
        .onAppear {
            startAnimation()
        }
        .accessibility(label: Text("Health improving with boost level \(Int(severity * 100)) percent"))
    }
    
    private func sparkle(at index: Int) -> some View {
        let angle = Double(index) * 120.0 // 120 degrees apart
        let radius: CGFloat = 25
        
        return Image(systemName: "sparkle")
            .font(.caption2)
            .foregroundColor(.yellow)
            .opacity(isAnimating ? 0.8 : 0.3)
            .scaleEffect(isAnimating ? 1.2 : 0.8)
            .offset(
                x: radius * cos(angle * .pi / 180),
                y: radius * sin(angle * .pi / 180)
            )
            .animation(
                Animation.easeInOut(duration: 1.5)
                    .repeatForever(autoreverses: true)
                    .delay(Double(index) * 0.2),
                value: isAnimating
            )
    }
    
    private func startAnimation() {
        withAnimation(Animation.easeInOut(duration: 1.0).repeatForever(autoreverses: true)) {
            isAnimating = true
            sparkleOffset = -8
            glowIntensity = 1.0 + severity
        }
    }
}

#Preview {
    VStack(spacing: 20) {
        LifeBoostView(severity: 0.3)
        LifeBoostView(severity: 0.7)
        LifeBoostView(severity: 1.0)
    }
    .padding()
    .background(.black.opacity(0.1))
}